function edgs = ceqspace(X, nb, D)

%CEQSPACE Edges for centered equi-spaced binning.
%
% INPUTS
% ------
%   X   - array of values to be discritized
%   NB  - number of bins
%   PAR - optional parameter
%
% USAGE
% -----
%
%   edgs = ceqspace(X, nb)
%   edgs = ceqspace(X, nb, D)
%
% edgs = ceqspace(X, nb) returns equispaced edges in the range [C-D,C+D], C
% being the mean of the values to be binned and D = max(C-m, M-C) and m and
% M being the minimum and maximum value in X, respectively.
%
% edgs = ceqspace(X, nb,D) returns equispaced edges in the range [C-D,C+D]
% where D is defined by the user.

% Copyright (C) 2010 Cesare Magri
% Version: 1.0.3

% -------
% LICENSE
% -------
% This software is distributed free under the condition that:
%
% 1. it shall not be incorporated in software that is subsequently sold;
%
% 2. the authorship of the software shall be acknowledged and the following
%    article shall be properly cited in any publication that uses results
%    generated by the software:
%
%      Magri C, Whittingstall K, Singh V, Logothetis NK, Panzeri S: A
%      toolbox for the fast information analysis of multiple-site LFP, EEG
%      and spike train recordings. BMC Neuroscience 2009 10(1):81;
%
% 3.  this notice shall remain in place in each source file.

% ----------
% DISCLAIMER
% ----------
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
% IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
% THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
% PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
% CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
% EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
% PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
% PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
% LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
% NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
% SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

meanX = mean(X(:));

minX  = min(X(:));
maxX  = max(X(:));

if nargin==3
    if isscalar(D)
        
        if minX<meanX-D || maxX>meanX+D
            msg = 'Values out of range provided.';
            error('ceqspace:XoutOfInterval', msg);
        end
    else
        msg = 'Invalid parameter.';
        error('ceqspace:invalidParameter', msg);
    end
else
    D = max(meanX-minX, maxX-meanX);
end

leftEdg  = meanX - D;
rightEdg = meanX + D;

edgs = linspace(leftEdg, rightEdg, nb+1);
% Stretching last edge to accomodate maxX:
edgs(nb+1) = edgs(nb+1) + 1;